<?php

namespace App\Filament\Resources;

use App\Filament\Resources\PackageResource\Pages;
use App\Models\Package;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Support\Str;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Tabs;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Get;
use Filament\Forms\Set;

class PackageResource extends Resource
{
    protected static ?string $model = Package::class;

    protected static ?string $navigationIcon = 'heroicon-o-credit-card';
    protected static ?string $navigationLabel = 'باقات الاشتراك';
    protected static ?string $modelLabel = 'باقة';
    protected static ?int $navigationSort = 4; // ترتيبها في القائمة

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Tabs::make('Package Configuration')
                    ->tabs([
                        // ------------------------------------------------
                        // التبويب الأول: المعلومات الأساسية والتسعير
                        // ------------------------------------------------
                        Tabs\Tab::make('معلومات الباقة')
                            ->icon('heroicon-o-information-circle')
                            ->schema([
                                Section::make('التفاصيل العامة')
                                    ->schema([
                                        TextInput::make('name')
                                            ->label('اسم الباقة')
                                            ->required()
                                            ->placeholder('مثلاً: الباقة الذهبية')
                                            ->live(onBlur: true)
                                            ->afterStateUpdated(function (Get $get, Set $set, ?string $old, ?string $state) {
                                                if (($get('slug') ?? '') !== Str::slug($old)) {
                                                    return;
                                                }
                                                $set('slug', Str::slug($state));
                                            }),

                                        TextInput::make('slug')
                                            ->label('الاسم البرمجي (Slug)')
                                            ->required()
                                            ->unique(ignoreRecord: true)
                                            ->helperText('يستخدم في الروابط والبرمجة'),

                                        Textarea::make('description')
                                            ->label('وصف الباقة')
                                            ->rows(3)
                                            ->columnSpanFull(),
                                    ])->columns(2),

                                Section::make('التسعير والعرض')
                                    ->schema([
                                        TextInput::make('price')
                                            ->label('السعر')
                                            ->numeric()
                                            ->prefix('$') // أو ريال
                                            ->required(),

                                        TextInput::make('duration_days')
                                            ->label('مدة الباقة (بالأيام)')
                                            ->numeric()
                                            ->default(30)
                                            ->helperText('30 = شهر، 365 = سنة'),

                                        TextInput::make('sort_order')
                                            ->label('ترتيب العرض')
                                            ->numeric()
                                            ->default(0),

                                        Toggle::make('is_visible')
                                            ->label('إظهار في الموقع')
                                            ->default(true),

                                        Toggle::make('is_popular')
                                            ->label('تمييز كـ "أفضل خيار"')
                                            ->onColor('warning'),
                                    ])->columns(3),
                            ]),

                        // ------------------------------------------------
                        // التبويب الثاني: الحدود (Limits / Quotas)
                        // ------------------------------------------------
                        Tabs\Tab::make('الحدود والأرصدة')
                            ->icon('heroicon-o-chart-bar')
                            ->schema([
                                Section::make('سياسات الاستخدام العادل')
                                    ->description('ضع القيمة 0 لجعل الميزة "غير محدودة"')
                                    ->schema([
                                        TextInput::make('limit_messages')
                                            ->label('رصيد الرسائل (شهري)')
                                            ->numeric()
                                            ->suffix('رسالة')
                                            ->default(1000),

                                        TextInput::make('limit_contacts')
                                            ->label('أقصى عدد جهات اتصال')
                                            ->numeric()
                                            ->suffix('جهة')
                                            ->default(500),

                                        TextInput::make('limit_campaigns')
                                            ->label('عدد الحملات الشهرية')
                                            ->numeric()
                                            ->default(5),

                                        TextInput::make('limit_wa_accounts')
                                            ->label('أرقام الواتساب المربوطة')
                                            ->numeric()
                                            ->default(1),

                                        TextInput::make('limit_team_members')
                                            ->label('عدد أعضاء الفريق الإضافيين')
                                            ->numeric()
                                            ->default(0),

                                        TextInput::make('limit_storage_mb')
                                            ->label('مساحة التخزين (MB)')
                                            ->numeric()
                                            ->default(100),
                                    ])->columns(3),
                            ]),

                        // ------------------------------------------------
                        // التبويب الثالث: الميزات التقنية (Features)
                        // ------------------------------------------------
                        Tabs\Tab::make('الميزات والصلاحيات')
                            ->icon('heroicon-o-star')
                            ->schema([
                                Section::make('تفعيل/تعطيل الخصائص')
                                    ->description('حدد الأدوات التي يحق للمشترك استخدامها')
                                    ->schema([
                                        Toggle::make('has_api_access')
                                            ->label('الوصول للـ API'),

                                        Toggle::make('has_chatbot')
                                            ->label('نظام الرد الآلي (Chatbot)'),

                                        Toggle::make('has_media_upload')
                                            ->label('إرسال الوسائط (صور/فيديو)'),

                                        Toggle::make('has_schedule_campaigns')
                                            ->label('جدولة الحملات مستقبلاً'),

                                        Toggle::make('has_excel_export')
                                            ->label('تصدير التقارير (Excel)'),

                                        Toggle::make('remove_branding')
                                            ->label('إزالة الحقوق (White Label)'),

                                        Toggle::make('has_priority_support')
                                            ->label('دعم فني ذو أولوية (VIP)'),
                                    ])->columns(2),
                            ]),
                    ])->columnSpanFull(),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('name')
                    ->label('اسم الباقة')
                    ->searchable()
                    ->weight('bold'),

                Tables\Columns\TextColumn::make('price')
                    ->label('السعر')
                    ->money('USD') // يمكنك تغيير العملة
                    ->sortable(),

                Tables\Columns\TextColumn::make('limit_messages')
                    ->label('رصيد الرسائل')
                    ->formatStateUsing(fn (string $state): string => $state == 0 ? 'غير محدود' : number_format($state)),

                Tables\Columns\IconColumn::make('is_visible')
                    ->label('نشطة')
                    ->boolean(),

                Tables\Columns\IconColumn::make('is_popular')
                    ->label('مفضلة')
                    ->boolean()
                    ->trueIcon('heroicon-s-star')
                    ->falseIcon('heroicon-o-star')
                    ->color('warning'),

                Tables\Columns\TextColumn::make('created_at')
                    ->label('تاريخ الإنشاء')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListPackages::route('/'),
            'create' => Pages\CreatePackage::route('/create'),
            'edit' => Pages\EditPackage::route('/{record}/edit'),
        ];
    }
}