<?php

namespace App\Filament\Resources;

use App\Filament\Resources\UserResource\Pages;
use App\Models\User;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Hash;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Components\Textarea;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Columns\ToggleColumn;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Filters\TernaryFilter;

class UserResource extends Resource
{
    protected static ?string $model = User::class;

    protected static ?string $navigationIcon = 'heroicon-o-user-group';
    protected static ?string $navigationLabel = 'إدارة العملاء';
    protected static ?string $modelLabel = 'عميل';
    protected static ?int $navigationSort = 1;

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                // --- القسم 1: المعلومات الشخصية ---
                Section::make('البيانات الأساسية')
                    ->description('معلومات الدخول والهوية')
                    ->schema([
                        TextInput::make('name')
                            ->label('اسم العميل')
                            ->required()
                            ->maxLength(255),

                        TextInput::make('email')
                            ->label('البريد الإلكتروني')
                            ->email()
                            ->required()
                            ->unique(ignoreRecord: true)
                            ->maxLength(255),

                        TextInput::make('password')
                            ->label('كلمة المرور')
                            ->password()
                            ->dehydrateStateUsing(fn ($state) => Hash::make($state))
                            ->dehydrated(fn ($state) => filled($state)) // لا يحدثها إذا تركت فارغة
                            ->required(fn (string $context): bool => $context === 'create'),
                    ])->columns(2),

                // --- القسم 2: تفاصيل الاشتراك والباقة ---
                Section::make('الاشتراك والباقة')
                    ->description('التحكم في باقة العميل وصلاحيته')
                    ->schema([
                        Select::make('package_id')
                            ->label('الباقة المشترك بها')
                            ->relationship('package', 'name')
                            ->preload()
                            ->required()
                            ->live() // تحديث مباشر
                            ->afterStateUpdated(function ($state, Forms\Set $set) {
                                // عند اختيار باقة، نضبط تاريخ الانتهاء تلقائياً حسب مدة الباقة
                                if ($state) {
                                    $package = \App\Models\Package::find($state);
                                    if ($package) {
                                        $set('subscription_ends_at', now()->addDays($package->duration_days));
                                        $set('credits', $package->limit_messages);
                                    }
                                }
                            }),

                        DatePicker::make('subscription_ends_at')
                            ->label('تاريخ انتهاء الاشتراك')
                            ->native(false)
                            ->displayFormat('d/m/Y')
                            ->required(),

                        TextInput::make('credits')
                            ->label('الرصيد المتاح (رسائل)')
                            ->numeric()
                            ->default(0)
                            ->helperText('يمكنك تعديل الرصيد يدوياً كبونص'),

                        Toggle::make('is_active')
                            ->label('حساب نشط')
                            ->onColor('success')
                            ->offColor('danger')
                            ->default(true)
                            ->helperText('عند التعطيل لن يتمكن العميل من الدخول'),

                    ])->columns(2),

                // --- القسم 3: ملاحظات الإدارة ---
                Section::make('ملاحظات إدارية')
                    ->collapsed()
                    ->schema([
                        Textarea::make('notes')
                            ->label('ملاحظات داخلية')
                            ->rows(3)
                            ->placeholder('اكتب أي ملاحظات عن العميل لا يراها إلا الإدارة...'),
                    ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('name')
                    ->label('العميل')
                    ->searchable()
                    ->sortable()
                    ->description(fn (User $record) => $record->email),

                // عرض الباقة كشعار ملون
                TextColumn::make('package.name')
                    ->label('الباقة')
                    ->badge()
                    ->color(fn ($state) => match ($state) {
                        'Basic' => 'gray',
                        'Pro' => 'warning',
                        'Enterprise' => 'success',
                        default => 'primary',
                    })
                    ->sortable(),

                TextColumn::make('credits')
                    ->label('الرصيد')
                    ->icon('heroicon-m-banknotes')
                    ->sortable(),

                // حالة الاشتراك (منتهي أم ساري)
                TextColumn::make('subscription_ends_at')
                    ->label('الانتهاء')
                    ->date('d/m/Y')
                    ->sortable()
                    ->description(fn (User $record) => $record->hasActiveSubscription() 
                        ? 'ساري المفعول' 
                        : 'منتهي الصلاحية')
                    ->color(fn (User $record) => $record->hasActiveSubscription() ? 'success' : 'danger'),

                // زر تفعيل/تعطيل سريع في الجدول مباشرة
                ToggleColumn::make('is_active')
                    ->label('الحالة')
                    ->onColor('success')
                    ->offColor('danger'),

                TextColumn::make('created_at')
                    ->label('تاريخ التسجيل')
                    ->date()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                // فلتر حسب الباقة
                SelectFilter::make('package')
                    ->label('تصفية حسب الباقة')
                    ->relationship('package', 'name'),

                // فلتر حسب الحالة (نشط/محظور)
                TernaryFilter::make('is_active')
                    ->label('حالة الحساب')
                    ->trueLabel('نشط فقط')
                    ->falseLabel('محظور فقط'),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                
                // زر سريع لتجديد الاشتراك شهر إضافي
                Tables\Actions\Action::make('renew')
                    ->label('تجديد شهر')
                    ->icon('heroicon-o-arrow-path')
                    ->color('success')
                    ->requiresConfirmation()
                    ->action(function (User $record) {
                        $record->update([
                            'subscription_ends_at' => \Carbon\Carbon::parse($record->subscription_ends_at)->addDays(30)
                        ]);
                        \Filament\Notifications\Notification::make()
                            ->title('تم التجديد بنجاح')
                            ->success()
                            ->send();
                    }),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            // يمكن هنا إضافة علاقة الحملات لرؤية حملات العميل
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListUsers::route('/'),
            'create' => Pages\CreateUser::route('/create'),
            'edit' => Pages\EditUser::route('/{record}/edit'),
        ];
    }
}