<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\WhatsAppAccount;
use App\Services\WhatsAppService;
use Illuminate\Support\Facades\Log;

class ExternalSendController extends Controller
{
    public function send(Request $request)
    {
        // 1. التحقق من المدخلات
        $request->validate([
            'to' => 'required|string',
            'type' => 'nullable|in:text,template',
            'device_id' => 'nullable|exists:whatsapp_accounts,id',
        ]);

        $type = $request->input('type', 'text');

        // تحقق إضافي حسب نوع الرسالة
        if ($type === 'text') {
            $request->validate(['text_message' => 'required|string']);
        } else {
            $request->validate([
                'template_name' => 'required|string',
                'language_code' => 'nullable|string',
                'components' => 'nullable|array'
            ]);
        }

        // الحصول على المستخدم من التوكن
        $user = $request->user();

        // 2. التحقق من الرصيد
        if ($user->credits < 1) {
            return response()->json([
                'status' => false,
                'message' => 'Insufficient credits'
            ], 402);
        }

        // 3. تحديد جهاز الإرسال
        $account = null;
        if ($request->device_id) {
            $account = WhatsAppAccount::where('id', $request->device_id)->where('user_id', $user->id)->first();
        } else {
            // اختيار أول جهاز متصل للمستخدم
            $account = WhatsAppAccount::where('user_id', $user->id)->first();
        }

        if (!$account) {
            return response()->json([
                'status' => false,
                'message' => 'No active WhatsApp account found'
            ], 404);
        }

        // 4. عملية الإرسال
        // ملاحظة: يفضل استخدام DB Transaction هنا لضمان سلامة البيانات
        $user->decrement('credits'); 

        $waService = new WhatsAppService();
        $result = false;
        $phone = $request->to;

        try {
            if ($type === 'text') {
                // إرسال رسالة نصية
                $result = $waService->sendText(
                    $account->phone_number_id,
                    $account->access_token,
                    $phone,
                    $request->text_message,
                    $user // 🔑 [التصحيح النهائي] تمرير كائن المستخدم كما يطلبه السيرفس
                );
            } else {
                // إرسال قالب
                $result = $waService->sendTemplate(
                    $account->phone_number_id,
                    $account->access_token,
                    $phone,
                    $request->template_name,
                    $request->language_code ?? 'en',
                    $request->components ?? []
                    // تأكد إذا كانت دالة sendTemplate تحتاج أيضاً للمستخدم كمعامل أخير أم لا
                );
            }
        } catch (\Exception $e) {
            Log::error("API Send Error: " . $e->getMessage());
            $result = false;
        }

        // 5. الرد النهائي
        if ($result) {
            return response()->json([
                'status' => true,
                'message' => 'Message sent successfully',
                'data' => [
                    // محاولة جلب ID الرسالة إذا كان متاحاً في الاستجابة
                    'id' => $result['messages'][0]['id'] ?? 'N/A',
                ],
                'remaining_credits' => $user->credits
            ], 200);
        } else {
            // إعادة الرصيد في حال الفشل
            $user->increment('credits');
            
            return response()->json([
                'status' => false,
                'message' => 'Failed to send message via Meta API'
            ], 500);
        }
    }
}