<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class WebhookController extends Controller
{
    /**
     * يستقبل ويعالج الرسائل الواردة من واتساب (Webhook).
     */
    public function handleIncomingMessage(Request $request)
    {
        // 1. تأمين الويب هوك (التحقق من التوقيع)
        if (!$this->verifyWebhookSignature($request)) {
            Log::warning('Webhook: توقيع غير صالح.', ['payload' => $request->all()]);
            return response()->json(['status' => 'Unauthorized'], 401);
        }

        // 2. معالجة بيانات الحمولة (Payload)
        $payload = $request->all();

        // 🔑 التأكد من أن الحمولة تحتوي على رسالة واردة
        if (!empty($payload['entry'][0]['changes'][0]['value']['messages'])) {
            $messageData = $payload['entry'][0]['changes'][0]['value']['messages'][0];
            
            // 🔑 استخراج البيانات الأساسية
            $from = $messageData['from']; // رقم المرسل
            $type = $messageData['type']; // نوع الرسالة (text, button, interactive, etc.)
            
            // 3. تمريرها إلى منطق الشات بوت
            $this->processChatMessage($from, $messageData);

            return response()->json(['status' => 'OK'], 200);
        }

        // إرجاع استجابة ناجحة لأي إشعار غير متعلق برسالة
        return response()->json(['status' => 'Ignored'], 200);
    }
    
    /**
     * التحقق من توقيع الويب هوك (حسب منصة WhatsApp API المستخدمة).
     * 💡 يجب عليك استبدال هذا بمنطق التحقق الفعلي الخاص بمنصتك (Meta/Cloud API أو منصة خارجية).
     */
    private function verifyWebhookSignature(Request $request): bool
    {
        // ⚠️ هذا مثال بسيط. يجب تطبيق منطق التوقيع SHA256 الفعلي هنا.
        // if ($request->header('X-Hub-Signature-256') === 'your_secret_hash') {
        //     return true;
        // }
        // For now, we assume it's secure for local testing.
        return true; 
    }

    /**
     * منطق الشات بوت للرد على الرسائل.
     */
    private function processChatMessage(string $from, array $messageData): void
    {
        $messageBody = strtolower($messageData['text']['body'] ?? ''); // محتوى الرسالة
        
        // 🔑 ربط هذه الرسالة بحساب واتساب خاص بك (باستخدام رقم هاتف المستلم)
        // $whatsappAccount = WhatsAppAccount::where('phone_number', $messageData['recipient_id'])->first(); 

        // ---------------------------------------------
        // 🚀 منطق الشات بوت الاحترافي يبدأ هنا
        // ---------------------------------------------

        $responseText = "مرحباً! أنا بوت WhatsMark الذكي. كيف يمكنني مساعدتك اليوم؟";
        
        if (str_contains($messageBody, 'مرحبا') || str_contains($messageBody, 'سلام')) {
            $responseText = "أهلاً بك! يمكنك الرد بأحد الخيارات التالية:\n1. الأسعار\n2. الميزات\n3. التحدث مع ممثل.";
            
        } elseif (str_contains($messageBody, 'أسعار') || $messageBody === '1') {
            $responseText = "لدينا عدة باقات:\n- الباقة الأساسية: 50 ر.س/الشهر\n- الباقة المتقدمة: 150 ر.س/الشهر.\nلطلب باقة، يرجى زيارة الرابط: " . route('landing', ['#pricing']);
            
        } elseif (str_contains($messageBody, 'ميزات') || $messageBody === '2') {
            $responseText = "أبرز ميزاتنا: إرسال جماعي سريع، تخصيص قوالب، وتتبع الأداء بدقة.";
            
        } elseif (str_contains($messageBody, 'تحدث') || $messageBody === '3') {
            $responseText = "تم تحويلك إلى ممثل خدمة العملاء. يرجى الانتظار.";
            // 💡 هنا يمكنك وضع منطق إرسال إشعار للموظف
            
        } else {
            $responseText = "عذراً، لم أفهم طلبك. يرجى اختيار أحد الخيارات أو إرسال كلمة 'مساعدة'.";
        }
        
        // ---------------------------------------------
        // 4. إرسال الرد
        // ---------------------------------------------
        
        // 💡 يجب عليك استخدام دالة الإرسال التي أنشأناها مسبقاً
        $this->sendWhatsAppMessage($from, $responseText);
    }

    /**
     * 💡 دالة وهمية لإرسال رسالة واتساب.
     * يجب استبدالها بدالة الإرسال الفعلية التي تستخدمها للرسائل الصادرة.
     */
    private function sendWhatsAppMessage(string $to, string $body): void
    {
        // ⚠️ يجب استدعاء API الإرسال الخاص بك هنا
        Log::info("ChatBot Reply Sent:", [
            'to' => $to, 
            'body' => $body
        ]);
    }
}