<?php

namespace App\Jobs;

use App\Models\Campaign;
use App\Models\Contact;
use App\Models\CampaignLog; // استدعاء موديل السجلات
use App\Services\WhatsAppService;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

class SendCampaignMessage implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public $campaign;
    public $contact;

    /**
     * استقبال الحملة وجهة الاتصال لتنفيذ المهمة عليهما
     */
    public function __construct(Campaign $campaign, Contact $contact)
    {
        $this->campaign = $campaign;
        $this->contact = $contact;
    }

    /**
     * تنفيذ المهمة
     */
    public function handle()
    {
        // 1. تجهيز خدمة الواتساب
        $whatsappService = new WhatsAppService();

        // 2. محاولة الإرسال باستخدام بيانات حساب الحملة
        $result = $whatsappService->sendTemplate(
            $this->campaign->whatsappAccount->phone_number_id,
            $this->campaign->whatsappAccount->access_token,
            $this->contact->phone,
            $this->campaign->template_name,
            $this->campaign->template_language
        );

        // متغيرات لتخزين النتيجة في السجل
        $status = 'failed';
        $messageId = null;
        $errorReason = null;

        // 3. التحقق من النتيجة وتحديث العدادات
        if (isset($result['messages'][0]['id'])) {
            // --- حالة النجاح ---
            $status = 'sent';
            $messageId = $result['messages'][0]['id'];
            
            // زيادة عداد النجاح في الحملة (Increment آمنة مع الطوابير)
            $this->campaign->increment('success_count');
        } else {
            // --- حالة الفشل ---
            $status = 'failed';
            // تخزين الخطأ كـ JSON لسهولة قراءته لاحقاً
            $errorReason = json_encode($result, JSON_UNESCAPED_UNICODE); 
            
            // زيادة عداد الفشل
            $this->campaign->increment('failed_count');
        }

        // 4. حفظ العملية في سجل التقارير (Logs)
        CampaignLog::create([
            'campaign_id'  => $this->campaign->id,
            'phone'        => $this->contact->phone,
            'status'       => $status,
            'message_id'   => $messageId,
            'error_reason' => $errorReason,
        ]);
    }
}